MobSpells = LibStub("AceAddon-3.0"):NewAddon("MobSpells")

local AceGUI = LibStub("AceGUI-3.0")
local AceConfig = LibStub("AceConfig-3.0")
local AceConfigDialog = LibStub("AceConfigDialog-3.0")

local L = LibStub("AceLocale-3.0"):GetLocale("MobSpells")

LibStub("AceAddon-3.0"):EmbedLibraries(
	MobSpells,
	"AceEvent-3.0",
	"AceConsole-3.0",
	"AceHook-3.0"
)

local defaults = {
	global = {
		mobs = {}
	},
	profile = {
		showTooltips = true,
		recordNone = true,
		recordPvp = nil,
		recordArena = nil,
		recordParty = true,
		recordRaid = true,
		recordWhenSolo = true,
		recordWhenParty = true,
		recordWhenRaid = true,
		fiveman = true,
		fivemanHeroic = true,
		tenman = true,
		twentyfiveman = true,
		tenmanHeroic = true,
		twentyfivemanHeroic = true,
	},
}

local dungeonMap = { "fiveman", "fivemanHeroic" }
local raidMap = { "tenman", "twentyfiveman", "tenmanHeroic", "twentyfivemanHeroic" }

local mobs
local bit_band = _G.bit.band
local frame

-- Upvalues
local next = _G.next
local select = _G.select
local pairs = _G.pairs
local type = _G.type
local table_sort = _G.table.sort
--[[
local COMBATLOG_OBJECT_CONTROL_NPC = _G.COMBATLOG_OBJECT_CONTROL_NPC
local COMBATLOG_OBJECT_TYPE_NPC = _G.COMBATLOG_OBJECT_TYPE_NPC
local COMBATLOG_OBJECT_REACTION_HOST = _G.COMBATLOG_OBJECT_REACTION_HOST
local COMBATLOG_OBJECT_REACTION_HOSTILE = _G.COMBATLOG_OBJECT_REACTION_HOSTILE
local COMBATLOG_DEFAULT_COLORS = _G.COMBATLOG_DEFAULT_COLORS
]]--
local tonumber = _G.tonumber
local UnitGUID = _G.UnitGUID
local GetZoneText = _G.GetZoneText
local GetTime = _G.GetTime
local UnitName = _G.UnitName
local tremove = _G.tremove

BINDING_NAME_MOBSPELLS = L["Show MobSpells"]
BINDING_HEADER_MOBSPELLS = L["Mob Spells"]

local function shouldLog()
	local db = MobSpells.db.profile
	local should = nil
	local i, t = IsInInstance()
	local d = GetInstanceDifficulty()
	if t == "none" then
		should = db.recordNone
	elseif t == "pvp" then
		should = db.recordPvp
	elseif t == "arena" then
		should = db.recordArena
	elseif t == "party" then
		if db.recordParty then
			local d = GetDungeonDifficulty()
			if db[dungeonMap[d]] then
				should = true
			end
		else
			should = nil
		end
	elseif t == "raid" then
		if db.recordRaid then
			local d = GetRaidDifficulty()
			if db[raidMap[d]] then
				should = true
			end
		else
			should = nil
		end
	end

	if GetNumRaidMembers() > 0 then
		should = should and db.recordWhenRaid
	elseif GetNumPartyMembers() > 0 then
		should = should and db.recordWhenParty
	elseif db.recordWhenSolo then
		should = should and true
	end

	if should then
		MobSpells:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
	else
		MobSpells:UnregisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
	end
end

local function get(info) return MobSpells.db.profile[info[#info]] end
local function set(info, v) MobSpells.db.profile[info[#info]] = v; shouldLog() end

local options = {
	type = "group",
	handler = MobSpells,
	childGroups = "tab",
	get = get,
	set = set,
	args = {
		desc = {
			type = "description",
			name = L["description"],
			order = 1,
			fontSize = "medium",
		},
		show = {
			type = "execute",
			name = L["Show MobSpells"],
			desc = L["Show MobSpells"],
			func = "Show",
			order = 2,
			width = "full",
		},
		report = {
			type = "execute",
			name = L["Report target's spells"],
			desc = L["Report your target spells to the group"],
			func = "ReportTargetSpells",
			order = 3,
			width = "full",
		},
		showTooltips = {
			type = "toggle",
			name = L["Use Tooltips"],
			desc = L["Show spell info in tooltips"],
			set = function(info, v)
				MobSpells.db.profile.showTooltips = v
				if v then
					MobSpells:SecureHookScript(GameTooltip, "OnTooltipSetUnit")
					MobSpells:Print(L["Now showing abilities in tooltip"])
				else
					MobSpells:Unhook(GameTooltip, "OnTooltipSetUnit")
					MobSpells:Print(L["NOT showing abilities in tooltip"])
				end
			end,
			order = 4,
			width = "full",
		},
		config = {
			type = "execute",
			name = L["Config"],
			desc = L["Configure MobSpells"],
			func = function() InterfaceOptionsFrame_OpenToCategory(MobSpells.optFrame) end,
			guiHidden = true,
		},
		recordIn = {
			type = "group",
			name = L["Record in ..."],
			order = 5,
			args = {
				recordNone = {
					type = "toggle",
					name = L["Outside"],
					desc = L["Whether to record when you're not in a battleground, arena or PvE instance."],
					order = 11,
				},
				recordPvp = {
					type = "toggle",
					name = L["PvP battleground"],
					desc = L["Whether to record when you're in a battleground."],
					order = 12,
				},
				recordArena = {
					type = "toggle",
					name = L["Arena"],
					desc = L["Whether to record when you're in an arena battle."],
					order = 13,
				},
				recordParty = {
					type = "toggle",
					name = L["5-man instance"],
					desc = L["Whether to record when you're in a 5-man instance."],
					order = 14,
				},
				recordRaid = {
					type = "toggle",
					name = L["Raid instance"],
					desc = L["Whether to record when you're in a raid instance."],
					order = 15,
				},
			},
		},
		recordWhen = {
			type = "group",
			name = L["Record when ..."],
			order = 6,
			args = {
				recordWhenSolo = {
					type = "toggle",
					name = L["Solo"],
					desc = L["Whether to record when you're playing solo."],
					order = 21,
				},
				recordWhenParty = {
					type = "toggle",
					name = L["Party"],
					desc = L["Whether to record when you're in a 5-man group."],
					order = 22,
				},
				recordWhenRaid = {
					type = "toggle",
					name = L["Raid"],
					desc = L["Whether to record when you're in a raid group."],
					order = 23,
				},
			},
		},
		recordDifficulty = {
			type = "group",
			name = L["Difficulty ..."],
			order = 7,
			args = {
				fiveman = {
					type = "toggle",
					name = _G["DUNGEON_DIFFICULTY1"],
					desc = L["Record in %s."]:format(_G["DUNGEON_DIFFICULTY1"]),
					order = 1,
				},
				fivemanHeroic = {
					type = "toggle",
					name = _G["DUNGEON_DIFFICULTY2"],
					desc = L["Record in %s."]:format(_G["DUNGEON_DIFFICULTY2"]),
					order = 2,
				},
				tenman = {
					type = "toggle",
					name = _G["RAID_DIFFICULTY1"],
					desc = L["Record in %s."]:format(_G["RAID_DIFFICULTY1"]),
					order = 3,
				},
				twentyfiveman = {
					type = "toggle",
					name = _G["RAID_DIFFICULTY2"],
					desc = L["Record in %s."]:format(_G["RAID_DIFFICULTY2"]),
					order = 4,
				},
				tenmanHeroic = {
					type = "toggle",
					name = _G["RAID_DIFFICULTY3"],
					desc = L["Record in %s."]:format(_G["RAID_DIFFICULTY3"]),
					order = 5,
				},
				twentyfivemanHeroic = {
					type = "toggle",
					name = _G["RAID_DIFFICULTY4"],
					desc = L["Record in %s."]:format(_G["RAID_DIFFICULTY4"]),
					order = 6,
				},
			},
		},
	},
}

AceConfig:RegisterOptionsTable("MobSpells", options, { "mobspells", "ms" } )

local function getZone()
	local zone = GetZoneText()
	local i, t = IsInInstance()
	if i and (t == "raid" or t == "party") then
		local diff = t == "raid" and _G["RAID_DIFFICULTY".. GetRaidDifficulty()] or _G["DUNGEON_DIFFICULTY".. GetDungeonDifficulty()]
		if not diff then diff = UNKNOWN end
		zone = zone .. " [" .. diff .. "]"
	end
	if not zone or zone:trim():len() == 0 then zone = UNKNOWN end
	return zone
end

function MobSpells:OnInitialize()
	self.db = LibStub("AceDB-3.0"):New("MobSpellsDB", defaults, true)

	mobs = self.db.global.mobs

	-- Upgrade old data
	for zone, v in pairs(mobs) do
		for mobID, mobData in pairs(v) do
			if mobID == 0 then mobs[zone][mobID]["name"] = UNKNOWN end -- hack to fix broken log
			if mobData.spells then
				for spellID, spellData in pairs(mobData.spells) do
					if type(spellData) == "table" then
						spellData.events = nil
					end
					-- Don't want to do this, I think. Mobs may have different behaviors for the same spell ID
					--[[
					if tonumber(spellID) ~= 0 and type(spellData) == "table" and not spells[spellID] then
						spells[spellID] = spellData
					end
					if tonumber(spellID) ~= 0 then
						mobData.spells[spellID] = true
					end
					]]--
				end
			end
		end
		if zone:trim():len() == 0 then
			mobs[UNKNOWN] = mobs[UNKNOWN] or {}
			for mobID, mobData in pairs(v) do
				mobs[UNKNOWN][mobID] = mobData
			end
			mobs[zone] = nil
		end
	end

	self:RegisterChatCommand("spells", "GetTargetSpells")
	self.searchIndex = self.searchIndex or "NPC"
	self.optFrame = AceConfigDialog:AddToBlizOptions("MobSpells", "MobSpells")
end

function MobSpells:OnEnable()
	if self.db.profile.showTooltips then
		self:SecureHookScript(GameTooltip, "OnTooltipSetUnit")
	end
	self:RegisterEvent("ZONE_CHANGED", shouldLog)
	self:RegisterEvent("ZONE_CHANGED_INDOORS", shouldLog)
	self:RegisterEvent("ZONE_CHANGED_NEW_AREA", shouldLog)
	self:RegisterEvent("PLAYER_ENTERING_WORLD", shouldLog)
	self:RegisterEvent("PARTY_MEMBERS_CHANGED", shouldLog)
	self:RegisterEvent("RAID_ROSTER_UPDATE", shouldLog)
end

--event which indicate the cast of a spell
--one cast of a spell should not produce more than one of these events
local start_events = {
	SWING_DAMAGE = true,
	SWING_MISSED = true,
	SPELL_CAST_START = true,
	SPELL_CAST_SUCCESS = true,
	SPELL_PERIODIC_CAST_START = true,
	SPELL_AURA_APPLIED = true,
}

local amount_suffixes = {
	["ENERGIZE"] = true,
	["LEECH"] = true,
	["DRAIN"] = true,
}

-- Compared to CombatLog_String_SchoolString, which tells you that the spell school is
-- 'Twilight', 'Divine', or whatever, this table holds the /actual/ schools, which
-- would be Arcane/Holy and Shadow/Holy, respectively.
local schools = {
	[1] = "Physical",
	[2] = "Holy",
	[4] = "Fire",
	[8] = "Nature",
	[16] = "Frost",
	[32] = "Shadow",
	[64] = "Arcane",
	[3] = "Holy/Physical",
	[5] = "Fire/Physical",
	[6] = "Holy/Fire",
	[9] = "Nature/Physical",
	[10] = "Nature/Holy",
	[12] = "Nature/Fire",
	[17] = "Frost/Physical",
	[18] = "Frost/Holy",
	[20] = "Frost/Fire",
	[24] = "Frost/Nature",
	[33] = "Shadow/Physical",
	[34] = "Shadow/Holy",
	[36] = "Shadow/Fire",
	[40] = "Shadow/Nature",
	[48] = "Shadow/Frost",
	[65] = "Arcane/Physical",
	[66] = "Arcane/Holy",
	[68] = "Arcane/Fire",
	[72] = "Arcane/Nature",
	[80] = "Arcane/Frost",
	[96] = "Arcane/Shadow",
	[28] = "Frost/Nature/Fire",
	[124] = "Arcane/Shadow/Frost/Nature/Fire",
	[126] = "Arcane/Shadow/Frost/Nature/Fire/Holy",
	[127] = "Arcane/Shadow/Frost/Nature/Fire/Holy/Physical",
}

local SPELL_BLACKLIST = {
	[1604] = true, 		--dazed
	[50424] = true, -- Mark of Blood effect
}

local MOB_BLACKLIST = {
	[33988] = true,
	[33136] = true,
}

--[[
	Checks if source or target of the event is a mob we are interested in.
	Returns:
	    the NPCID of the mob or nil
	    the GUID of the mob
	    boolean flag indicating if the mob is destination(true) or source(false) of the event
	    the name of the mob
	]]
function MobSpells:GetMobID(eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags)
	local guid, name, isDest

	if bit_band(srcFlags, COMBATLOG_OBJECT_CONTROL_NPC) == COMBATLOG_OBJECT_CONTROL_NPC and
		bit_band(srcFlags, COMBATLOG_OBJECT_TYPE_NPC) == COMBATLOG_OBJECT_TYPE_NPC then

		isDest = false
		guid = srcGUID
		name = srcName
	elseif bit_band(dstFlags, COMBATLOG_OBJECT_CONTROL_NPC) == COMBATLOG_OBJECT_CONTROL_NPC and
		bit_band(dstFlags, COMBATLOG_OBJECT_REACTION_HOSTILE) == COMBATLOG_OBJECT_REACTION_HOSTILE and
		bit_band(dstFlags, COMBATLOG_OBJECT_TYPE_NPC) == COMBATLOG_OBJECT_TYPE_NPC then

		isDest = true
		guid = dstGUID
		name = dstName
	end

	if not guid then return end
	local NPCID = tonumber(guid:sub(7,10),16)
	return NPCID, guid, isDest, name
end

do
	local tmp = {}
	local DEFAULT_COLOR = {r = 1, g = 1, b = 1, a = 1}
	function MobSpells:OnTooltipSetUnit(tt)
		local zone = getZone()
		if not zone or not mobs[zone] then return end
		local guid = UnitGUID("mouseover")
		if not guid then return end
		local NPCID = tonumber(guid:sub(7,10),16)
		if NPCID == 0 then return end

		if mobs[zone][NPCID] and mobs[zone][NPCID].spells and not MOB_BLACKLIST[NPCID] then
			for k, v in pairs(mobs[zone][NPCID].spells) do
				if k ~= 0 then
					local name = GetSpellInfo(k)
					if name and not tmp[name] then
						local c = COMBATLOG_DEFAULT_COLORS.schoolColoring[v.school]
						if not c then
							c = COMBATLOG_DEFAULT_COLORS.schoolColoring.default or DEFAULT_COLOR
						end
						tmp[#tmp+1] = ("|c%02x%02x%02x%02x%s|r"):format(c.a * 255, c.r * 255, c.g * 255, c.b * 255, name)
						tmp[name] = true
					end
				end
			end
		end
		if #tmp > 0 then
			tt:AddLine(L["Casts"] .. ": " .. table.concat(tmp, ", "), 1, 1, 1, 1)
			wipe(tmp)
		end
	end
end

--suffixes of spells which affect other spells
local extra_spell_suffixes = {
	["INTERRUPT"] = true,
	["AURA_DISPELLED"] = true,
	["AURA_STOLEN"] = true,
}

--[[
	Checks if the events contains information on an spell we are interested in.
	Returns:
	    the spell ID of the spell of interest or nil
	    the start of the parameters of the spell-event
	    the prefix of the event
	    the suffix of the event
	]]
function MobSpells:GetSpellID(eventtype, isDest, ...)
	local spellID, school, _

	local prefix = eventtype:sub(1, 5)

	if prefix == "ENVIR" then return end
	if prefix == "SPELL" and eventtype:sub(7, 14) == "PERIODIC" then prefix = "SWING_PERIODIC" end

	local paramStart = 12
	if prefix == "SWING" then paramStart = 9 end

	local suffix = eventtype:sub(#prefix + 2)

	--if the mob is the destination of the spell we are only interested in it if the spell was a
	--spell which affected an aura of the mob
	if isDest and extra_spell_suffixes[suffix] then
		local spellIDTmp, _, schoolTmp, auraType = select(paramStart, ...)
		if auraType and auraType == "BUFF" then
			spellID = spellIDTmp
			school = schoolTmp
		end

	--otherwise everything might be interresting
	elseif not isDest then
		if prefix == "SWING" then
			spellID = 0
		else
			spellID, _, school = select(9, ...)
		end
	end

	return spellID, paramStart, prefix, suffix, school
end

local is41 = tonumber((select(2, GetBuildInfo()))) > 13681 -- XXX
function MobSpells:COMBAT_LOG_EVENT_UNFILTERED(event, ...)
	local timestamp, eventtype, _, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellID
	if is41 then
		timestamp, eventtype, _, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellID = ...
	else
		timestamp, eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, spellID = ...
	end

	--check if the events affects a mob we are interested in and return its guid (or nil)
	local NPCID, guid, isDest, name = self:GetMobID(eventtype, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags)
	if not NPCID or MOB_BLACKLIST[NPCID] then return end
	if NPCID == 0 then
		name = "Unknown"
	end

	--check if the events contains a spell we are interested in and return its id (or nil)
	local spellID, paramStart, prefix, suffix, school = self:GetSpellID(eventtype, isDest, ...)
	if not spellID or SPELL_BLACKLIST[spellID] then return end

	--ignore dispels, i.e. mobs breaking stealth
	if eventtype == "SPELL_AURA_BROKEN" or eventtype == "SPELL_AURA_BROKEN_SPELL" then return end

	--ignore Haunt heals, since blizzard chose to code it as victim casting on player
	if eventtype == "SPELL_HEAL" and spellID == 48210 then return end

	--get zone + difficulty
	local zone = getZone()
	if not zone then return end

	mobs[zone] = mobs[zone] or {}
	mobs[zone][NPCID] = mobs[zone][NPCID] or {name = name, spells = {}}
	local spell
	spell = mobs[zone][NPCID].spells[spellID] or {}
	mobs[zone][NPCID].spells[spellID] = spell

	if type(spell) ~= "table" then
		local school = spell
		spell = {}
		spell.school = school
	end

	-----------start of parsing spell informations-----------

	spell.school = school

	--flags concerning the ability of the player to manipualte the spell
	if not spell.interruptable and suffix == "INTERRUPT" then spell.interruptable = true end
	if not spell.stealable and suffix == "AURA_STOLEN" then spell.stealable = true end
	if not spell.dispellable and suffix == "AURA_DISPELLED" then spell.dispellable = true end

	--information on time between casts of a spell
	if start_events[eventtype] then
		local time = GetTime()
		if spell.lastTime and spell.lastGUID == guid then
			local gone = time - spell.lastTime

			if (not spell.minSpeed or spell.minSpeed > gone) and gone > 0.35 then
				spell.minSpeed = gone
			end
			if (not spell.maxSpeed or spell.maxSpeed < gone) and gone < 120 then
				spell.maxSpeed = gone
			end
		end
		spell.lastTime = time
		spell.lastGUID = guid
	end

	--informations concerning damage
	if suffix == "DAMAGE" then
		local amount, school, resisted, blocked, absorbed, critical, glancing, crushing = select(paramStart, ...)
		spell.schoolName = schools[school]

		--flags on reduce-ability of the damage (resist and block)
		if resisted then
			spell.resisted = true
			amount = amount + resisted
		end
		if blocked then
			spell.blocked = true
			amount = amount + blocked
		end
		if absorbed then amount = amount + absorbed end

		--flags on damage burst-ability
		if critical then spell.critical = true end
		if glancing then spell.glancing = true end
		if crushing then spell.crushing = true end

		--don't record bursted damage (for now)
		if not critical and not glancing and not crushing then
			local min = spell.amountMin
			if not min or min > amount then
			   spell.amountMin = amount
			end

			local max = spell.amountMax
			if not max or max < amount then
			   spell.amountMax = amount
			end
		end

	--information concerning heals
	elseif suffix == "HEAL" then
		local amount, critical = select(paramStart, ...)

		--don't record crit heals but set the critable-flag
		if critical then
			spell.critical = true
		else
			local min = spell.amountMin
			if not min or min > amount then
			   spell.amountMin = amount
			end

			local max = spell.amountMax
			if not max or max < amount then
			   spell.amountMax = amount
			end
		end

	--information concerning spell misses (avoidance flags)
	elseif suffix == "MISSED" then
		local missType = select(paramStart, ...)

		if missType == "DODGE" then
			spell.dodge = true
		elseif missType == "PARRY" then
			spell.parry = true
		elseif missType == "REFLECT" then
			spell.reflect = true
		elseif missType == "RESIST" then
			spell.resist = true
		elseif missType ~= "MISS" then
			spell.missing = true
		end

	--information on generel spells which have an amount parameter (leech, drain and energize)
	elseif amount_suffixes[suffix] then
		local amount, critical = select(paramStart, ...)

		local min = spell.amountMin
		if not min or min > amount then
		   spell.amountMin = amount
		end

		local max = spell.amountMax
		if not max or max < amount then
		   spell.amountMax = amount
		end
	end
end

local function showSpellTooltip(this)
	GameTooltip:ClearLines()
	GameTooltip:SetOwner(this.frame, "ANCHOR_RIGHT")
	if this:GetUserData("spellID") ~= 0 then
		GameTooltip:SetHyperlink(("spell:%s"):format(this:GetUserData("spellID")))
	else
		GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Melee Attack"]))
	end
	GameTooltip:AddLine(" ")
	if this:GetUserData("spellID") ~= 0 then GameTooltip:AddLine(("|cffffffff%s #|r|cffffff00%s|r"):format(L["Spell ID"], this:GetUserData("spellID"))) end

	local spell = this:GetUserData("spell")
	if spell.amountMin then
		GameTooltip:AddLine(("|cffffff00%s-%s %s|r"):format(spell.amountMin, spell.amountMax, spell.schoolName or ""))
	elseif spell.schoolName then
		GameTooltip:AddLine(("|cffffffff%s|r |cffffff00%s|r"):format(L["School"], spell.schoolName))
	end
	if spell.minSpeed and spell.maxSpeed then GameTooltip:AddLine(("|cffffffff%s|r |cffffff00%.1f-%.1f %s|r"):format(L["Every"], spell.minSpeed, spell.maxSpeed, L["seconds"])) end

	if spell.missing then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can miss"])) end
	if spell.resist then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can fully resist"])) end
	if spell.reflect then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can be reflected"])) end
	if spell.dodge then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can be dodged"])) end
	if spell.parry then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can be parried"])) end

	if spell.resisted then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can be resisted"])) end
	if spell.blocked then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can be blocked"])) end

	if spell.dispellable then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can be dispelled"])) end
	if spell.stealable then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can be stolen"])) end
	if spell.interruptable then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can be interrupted"])) end

	if spell.critical then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can crit"])) end
	if spell.glancing then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can crush"])) end
	if spell.crushing then GameTooltip:AddLine(("|cffffffff%s|r"):format(L["Can glance"])) end

	GameTooltip:Show()
end

local function hideTooltip()
	GameTooltip:Hide()
end

local addLink

local function SelectSpell(widget, event, value)
	widget:ReleaseChildren()

	local zone, mobID = ("\001"):split(value)

	if not mobID then return end
	local zoneData = mobs[zone]
	local data = zoneData[tonumber(mobID)]
	if not data then return end

	local sf = AceGUI:Create("ScrollFrame")
	sf:SetLayout("Flow")

	local heading1 = AceGUI:Create("Heading")
	heading1:SetText(zone.. ": " .. data.name .. " (" .. mobID .. ")")
	heading1.width = "fill"
	sf:AddChild(heading1)

	for v, spell in pairs(data.spells) do
		local school = spell.school
		local name, rank, icon
		if v == 0 then
			name = L["Melee Attack"]
			rank = nil
			icon = [[Interface\Icons\INV_SWORD_06]]
		else
			name, rank, icon = GetSpellInfo(v)
		end
		local sFrame = AceGUI:Create("SimpleGroup")

		local label = AceGUI:Create("InteractiveLabel")
		if rank and #rank > 0 then
			label:SetText((" [%s (%s)] |cff777777(%d)|r"):format(name, rank, v))
		else
			label:SetText((" [%s] |cff777777(%d)|r"):format(name, v))
		end
		local colors = COMBATLOG_DEFAULT_COLORS.schoolColoring[school] or COMBATLOG_DEFAULT_COLORS.schoolColoring.default
		if colors then
			label.label:SetTextColor(colors.r, colors.g, colors.b, colors.a)
		end
		label:SetWidth(300)
		local p, h, f = label.label:GetFont()
		label:SetFont("Fonts\\ARIALN.TTF", 15, f)
		label:SetImageSize(22, 22)
		label:SetUserData("spellID", v)
		label:SetUserData("spell", spell)
		label:SetUserData("zone", zone)
		label:SetUserData("mobID", mobID)
		label:SetCallback("OnEnter", showSpellTooltip)
		label:SetCallback("OnLeave", hideTooltip)
		label:SetCallback("OnClick", addLink)
		label:SetHighlight("Interface\\Buttons\\UI-PlusButton-Hilight")
		if icon then
			label:SetImage(icon)
		end

		sFrame:AddChild(label)
		sf:AddChild(label)
	end

	widget:AddChild(sf)
end

function MobSpells:CreateFrame()
	if frame then return end

	local f = AceGUI:Create("Frame")
	_G["MobSpellsFrame"] = f.frame
	UISpecialFrames[#UISpecialFrames+1] = "MobSpellsFrame"

	f:SetTitle(L["Mob Spells"])
	f:SetStatusText("")
	f:SetLayout("Flow")
	f:SetWidth(650)
	f:SetHeight(450)

	local dropdown = AceGUI:Create("Dropdown")
	dropdown:SetText(L["NPCs"])
	dropdown:SetLabel(L["Search"])
	dropdown:SetList({NPC = L["NPCs"], SPELL = L["Spells"]})
	dropdown:SetWidth(100)
	dropdown:SetCallback("OnValueChanged",function(widget,event,value) MobSpells.searchIndex = value end )

	local edit = AceGUI:Create("EditBox")
	edit:SetText("")
	edit:SetWidth(200)
	edit:SetLabel(L["Keywords"])
	edit:SetCallback("OnEnterPressed",function(widget,event,text) MobSpells:Filter(text) end )
	edit:SetCallback("OnTextChanged",function(widget,event,text) MobSpells:Filter(text) end )
	f.editBox = edit

	local tgtbutton = AceGUI:Create("Button")
	tgtbutton:SetText(L["Find Target"])
	tgtbutton:SetWidth(150)
	tgtbutton:SetCallback("OnClick", self.GetTargetSpells)

	local t = AceGUI:Create("TreeGroup")
	t:SetLayout("Fill")
	t:SetCallback("OnGroupSelected", SelectSpell)
	t:SetCallback("OnClick", self.OnTreeClick)
	t:SetFullWidth(true)
	t:SetFullHeight(true)

	f:AddChildren(dropdown, edit, tgtbutton, t)

	f.tree = t

	frame = f
end

do
	local menuFrame = CreateFrame("Frame", "MobSpellsMenu", UIParent, "UIDropDownMenuTemplate")
	local deleteEntry = function(_, uniq)
		local zone, mobID, spellID = ("\001"):split(uniq)
		spellID, mobID = tonumber(spellID), tonumber(mobID)
		if spellID then
			MobSpells:Print(L["Deleted "] .. (GetSpellInfo(spellID) or L["Melee Attack"]))
			mobs[zone][mobID].spells[spellID] = nil
			frame.tree:SelectByValue(zone .. "\001" .. mobID)
			local ct = 0
			for k, v in pairs(mobs[zone][mobID].spells) do
				ct = ct + 1
			end
			if ct == 0 then
				mobs[zone][mobID] = nil
				ct = 0
				for k, v in pairs(mobs[zone]) do
					ct = ct + 1
				end
				if ct == 0 then
					mobs[zone] = nil
				end
			end
		elseif mobID then
			MobSpells:Print(L["Deleted "] .. mobs[zone][mobID].name)
			mobs[zone][mobID] = nil

			local ct = 0
			for k, v in pairs(mobs[zone]) do
				ct = ct + 1
			end
			if ct == 0 then
				mobs[zone] = nil
			end
		else
			MobSpells:Print(L["Deleted "] .. zone)
			mobs[zone] = nil
		end
		if MobSpellsFrame:IsVisible() then
			frame.tree:SetTree(MobSpells:BuildTree())
		end
	end

	local reportEntry = function(_, uniq)
		local zone, mobID = ("\001"):split(uniq)
		mobID = tonumber(mobID)
		MobSpells:ReportMobSpells(zone, mobID)
	end

	local menuDelete = {
		{}
	}
	menuDelete[2] = { text = L["Cancel"], func = function() CloseDropDownMenus(1) end }

	function addLink(self)
		menuFrame:Hide()
		if GetMouseButtonClicked() == "RightButton" then
			menuDelete[1].text = L["Delete this spell"]
			menuDelete[1].func = deleteEntry
			menuDelete[1].arg1 = self:GetUserData("zone") .. "\001" .. self:GetUserData("mobID") .. "\001" .. self:GetUserData("spellID")
			EasyMenu(menuDelete, menuFrame, "cursor", nil, nil, "MENU")
		else
			if IsShiftKeyDown() and ChatFrame1EditBox:IsShown() then
				ChatFrame1EditBox:Insert(GetSpellLink(self:GetUserData("spellID")))
			end
		end
	end

	function MobSpells.OnTreeClick(widget, evt, unique, selected)
		menuFrame:Hide()
		if GetMouseButtonClicked() == "RightButton" then
			local zone, mobID = ("\001"):split(unique)
			menuDelete[1].arg1 = unique
			menuDelete[1].func = deleteEntry
			if mobID then
				menuDelete[1].text = L["Delete this mob"]
				menuDelete[2].text = L["Report this mob"]
				menuDelete[2].arg1 = unique
				menuDelete[2].func = reportEntry
			else
				menuDelete[1].text = L["Delete this zone"]
				menuDelete[2] = nil
			end
			EasyMenu(menuDelete, menuFrame, "cursor", nil, nil, "MENU")
		end
	end
end

local function comp(a, b)
	if a.text > b.text then
		return false
	elseif a.text < b.text then
		return true
	else
		if a.value > b.value then
			return false
		elseif a.value < b.value then
			return true
		else
			return false
		end
	end
end

local t = {}
function MobSpells:BuildTree()
	t = wipe(t)

	local zones = {}
	for zone in pairs(mobs) do
		zones[#zones+1] = zone
	end
	table_sort(zones)

	for i, zone in next, zones do
		local tzone = {text=zone,value=zone}
		local tmp = {}
		for id, data in pairs(mobs[zone]) do
			tmp[#tmp+1] = {text=data.name, value=id}
		end
		table_sort(tmp, comp)
		tzone.children = tmp
		t[#t+1] = tzone
	end
	zones = nil

	return t
end

function MobSpells:Filter(text)
	local t = self:BuildTree()
	-- Iterate zones
	local zOffset = 0
	for i = 1, #t do
		local v = t[i-zOffset]
		local offset = 0
		for j = 1, #v.children do
			local child = v.children[j-offset]
			local useMob = false
			if #text == 0 then
				useMob = true
			end
			if self.searchIndex == "NPC" and child.text:lower():match(text:lower()) then
				useMob = true
			elseif self.searchIndex == "SPELL" then
				for k, v in pairs(mobs[v.value][child.value].spells) do
					local name = GetSpellInfo(k)
					if name and name:lower():match(text:lower()) then
						useMob = true
						break
					end
				end
			end
			if not useMob then
				tremove(v.children, j - offset)
				offset = offset + 1
			end
		end
		if #v.children == 0 then
			tremove(t, i - zOffset)
			zOffset = zOffset + 1
		end
	end
	frame.tree:SetTree(t)
end

function MobSpells:GetSpells(unit)
	if not UnitExists(unit) then
		self:Print(L["No target selected"])
		return
	end
	local name = UnitName(unit)
	MobSpells.searchIndex = "NPC"
	self:Show()
	frame.editBox:SetText(name)
	self:Filter(name)
	local guid = UnitGUID(unit)
	local NPCID = tonumber(guid:sub(7,10),16)
	local zone = getZone()
	if not zone then return end

	frame.tree:SelectByValue(zone .. "\001" .. NPCID)
end

function MobSpells.GetTargetSpells()
	MobSpells:GetSpells(UnitExists("target") and "target" or "mouseover")
end

function MobSpells:ReportTargetSpells()
	local unit = UnitExists("target") and "target" or "mouseover"
	if not UnitExists(unit) then
		self:Print(L["No target selected"])
		return
	end
	local guid = UnitGUID(unit)
	local NPCID = tonumber(guid:sub(7,10),16)
	local name = UnitName(unit)
	local zone = getZone()
	if not zone then return end

	self:ReportMobSpells(zone, NPCID, name)
end

local function SendMsg(s)
	if GetNumRaidMembers() > 0 then
		SendChatMessage(s,"RAID")
	elseif GetNumPartyMembers() > 0 then
		SendChatMessage(s,"PARTY")
	else
		print(s)
	end
end

function MobSpells:ReportMobSpells(zone, NPCID, nameoptional)
	local name = nameoptional or (mobs[zone] and mobs[zone][NPCID] and mobs[zone][NPCID].name) or ""
	local s = name .. L[" casts: "]
	local added = false
	if mobs and mobs[zone] and mobs[zone][NPCID] and mobs[zone][NPCID].spells then
		for k, v in pairs(mobs[zone][NPCID].spells) do
			local link = GetSpellLink(k)
			if link and #link > 0 then
				if #s + #link > 255 then
					SendMsg(s)
					s = ""
				end
				s = s .. link
				added = true
			end
		end
		if #s > 0 and added then
			SendMsg(s)
		elseif not added then
			print(L["No recorded spells for "] .. name)
		end
	end
end
function MobSpells:Show()
	self:CreateFrame()
	frame.tree:SetTree(self:BuildTree())
	frame:Show()
end
function MobSpells:Toggle()
	if frame and frame:IsShown() then
		frame:Hide()
	else
		self:Show()
	end
end

