--[[
    Armory Addon for World of Warcraft(tm).
    Revision: 454 2011-11-12T19:14:55Z
    URL: http://www.wow-neighbours.com

    License:
        This program is free software; you can redistribute it and/or
        modify it under the terms of the GNU General Public License
        as published by the Free Software Foundation; either version 2
        of the License, or (at your option) any later version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program(see GPL.txt); if not, write to the Free Software
        Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

    Note:
        This AddOn's source code is specifically designed to work with
        World of Warcraft's interpreted AddOn system.
        You have an implicit licence to use this AddOn with these facilities
        since that is it's designated purpose as per:
        http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat
--]]

local Armory = Armory;
local container = "Glyphs";
local knownContainer = "Known";

----------------------------------------------------------
-- Glyphs Internals
----------------------------------------------------------

local function GetGlyphValue(key, talentGroup)
    return Armory.selectedDbBaseEntry:GetValue(container, talentGroup, key);
end

----------------------------------------------------------
-- Glyphs Storage
----------------------------------------------------------

function Armory:GlyphsExists()
    local dbEntry = self.playerDbBaseEntry;
    local activeTalentGroup = _G.GetActiveTalentGroup();
    return dbEntry:Contains(container, activeTalentGroup) and dbEntry:Contains(container, knownContainer);
end

function Armory:ClearGlyphs()
    local dbEntry = self.selectedDbBaseEntry;
    dbEntry:SetValue(container, nil);
end

function Armory:UpdateGlyphs()
    local dbEntry = self.playerDbBaseEntry;
    local activeTalentGroup = _G.GetActiveTalentGroup();

    if ( not self:HasSpellBook() or _G.UnitLevel("player") < SHOW_INSCRIPTION_LEVEL ) then
        dbEntry:SetValue(container, nil);
        return;
    end
    
    if ( not self:IsLocked(container) ) then
        self:Lock(container);

        self:PrintDebug("UPDATE", container);

        for talentGroup = 1, MAX_TALENT_GROUPS do
            if ( talentGroup == activeTalentGroup or not dbEntry:Contains(container, talentGroup) ) then
                self:UpdateGlyphGroup(dbEntry, talentGroup);
            end
        end

        self:UpdateKnownGlyphs(dbEntry);
        
        self:Unlock(container);
    else
        self:PrintDebug("LOCKED", container);
    end
end

function Armory:UpdateGlyphGroup(dbEntry, talentGroup)
    dbEntry:SelectContainer(container, talentGroup);
    for i = 1, ARMORY_NUM_GLYPH_SLOTS do
        if ( _G.GetGlyphSocketInfo(i, talentGroup) ) then
            dbEntry:SetValue(3, container, talentGroup, "Info"..i, _G.GetGlyphSocketInfo(i, talentGroup));
            dbEntry:SetValue(3, container, talentGroup, "Link"..i, _G.GetGlyphLink(i, talentGroup));
        else
            dbEntry:SetValue(3, container, talentGroup, "Info"..i, nil);
            dbEntry:SetValue(3, container, talentGroup, "Link"..i, nil);
        end
    end
end

function Armory:UpdateKnownGlyphs(dbEntry)
    local index = 1;
    local name, isKnown, link;
    
    while ( _G.GetGlyphInfo(index) ) do
        name, _, isKnown, _, _, link = _G.GetGlyphInfo(index);
        if ( name ~= "header" ) then
            name = self:GetNameFromLink(link);
            if ( name ) then
                dbEntry:SetValue(3, container, knownContainer, name, isKnown and true or false, link);
            end
        end
        index = index + 1;
    end
end

----------------------------------------------------------
-- Glyphs Interface
----------------------------------------------------------

function Armory:GetGlyphSocketInfo(id, talentGroup)
    return GetGlyphValue("Info"..id, talentGroup);
end

function Armory:GetGlyphLink(id, talentGroup)
    return GetGlyphValue("Link"..id, talentGroup);
end

----------------------------------------------------------
-- Find Methods
----------------------------------------------------------

function Armory:FindGlyphs(...)
    local dbEntry = self.selectedDbBaseEntry;
    local list = {};
    
    if ( self:UnitLevel("player") >= SHOW_INSCRIPTION_LEVEL ) then
        local glyphs = dbEntry:GetValue(container, knownContainer);
        if ( glyphs ) then
            local unknown = strlower(select(1, ...)) == strlower(UNKNOWN) or select(1, ...) == "?";
            local isKnown, link, text, found, crafters, label;
            for name in pairs(glyphs) do
                isKnown, link = dbEntry:GetValue(container, knownContainer, name);
                if ( unknown ) then
                    found = not isKnown;
                else
                    if ( link and self:GetConfigExtendedSearch() ) then
                        text = self:GetTextFromLink(link);
                    else
                        text = name;
                    end
                    found = self:FindTextParts(text, ...);
                end
                if ( found ) then
                    if ( isKnown ) then
                        label = GLYPHS;
                        name = self:HexColor(self:GetConfigKnownColor())..name;
                    else
                        crafters = self:GetInscribers(name, self:UnitClass("player"));
                        if ( #crafters > 0 ) then
                            label = strjoin(", ", unpack(crafters));
                            name = self:HexColor(self:GetConfigCanLearnColor())..name;
                        else
                            label=OTHER;
                        end
                    end
                    table.insert(list, {label=label, name=name, link=link});
                end
            end
        end
    end
        
    return list;
end

local knownBy = {};
local function AddKnownBy(name)
    if ( Armory:GetConfigShowKnownBy() and name ~= Armory.player ) then
        table.insert(knownBy, name);
    end
end

local canLearn = {};
local function AddCanLearn(name)
    if ( Armory:GetConfigShowCanLearn() ) then
        table.insert(canLearn, name);
    end
end

local hasSkill = {};
local function AddHasSkill(name)
    if ( Armory:GetConfigShowHasSkill() ) then
        table.insert(hasSkill, name);
    end
end

function Armory:GetGlyphAltInfo(name, reqClass, reqLevel)
    table.wipe(knownBy);
    table.wipe(hasSkill);
    table.wipe(canLearn);

    if ( self:HasSpellBook() ) then
        local currentProfile = self:CurrentProfile();

        for _, character in ipairs(self:CharacterList(self.playerRealm)) do
            self:LoadProfile(self.playerRealm, character);
            
            local class = self:UnitClass("player");
            if ( reqClass == class ) then
                local dbEntry = self.selectedDbBaseEntry;
                local known = dbEntry:GetValue(container, knownContainer, name)
                
                if ( known ) then
                    AddKnownBy(character);
                else
                    local learnable = self:UnitLevel("player") >= reqLevel;
                    local attainable = not learnable;
                    local unknown = not dbEntry:Contains(container, knownContainer, name);
                    
                    if ( unknown ) then
                        character = character.." (?)";
                    end
                    if ( attainable ) then
                        AddHasSkill(character);
                    elseif ( learnable ) then
                        AddCanLearn(character);
                    end
                end
            end
        end
        self:SelectProfile(currentProfile);
    end

    return knownBy, hasSkill, canLearn;
end
